package com.rameses.rcp.common;

import com.rameses.rcp.annotations.*;
import com.rameses.osiris2.client.*;
import com.rameses.osiris2.common.*;
import com.rameses.io.IOStream;
import com.rameses.util.ExprVar;

public class FileDownloader extends AbstractFileDownloader {
    
    @Service( dynamic = true ) 
    def dynaSvc; 
    
    def _param;
    def _entity;

    String connection; 
    
    public FileDownloader() {
    }

    public FileDownloader( Map param ) {
        if ( !param?.objid ) {
            throw new Exception('objid  parameter is required in FileDownloader');
        }
        this._param = param; 
        if ( param.connection instanceof String ) {
            this.connection = param.connection; 
        }
    }
 
    public void setConnection( String name ) {
        this.connection = name; 
    }
    
    def getService() {
        def fm = com.rameses.filemgmt.FileManager.instance;
        def db = fm.getDbProvider(); 
        return db.getService( connection );
    }
    
    public def open() {
        _entity = getService().readItem([ objid: _param.objid ]);
        if ( _entity ) {
            def sb = new StringBuilder();
            sb.append( _entity.objid );
            if ( _entity.filetype ) {
                sb.append(".");
                sb.append( _entity.filetype );
            }
            _entity.filename = sb.toString();
        }
        return _entity;
    }

    public def getData() {
        return _entity; 
    }
    
    public java.net.URL createURL() {
        def item = getData();
        if ( item == null ) item = open();
        if ( !item ) return null; 
        
        item._validated = true; 
        return createURL( item );
    }

    final void validateItem( Map item ) {
        if ( item == null ) 
            throw new IllegalArgumentException("item argument is required in validateItem");

        if ( dynaSvc == null ) {
            def fm = com.rameses.filemgmt.FileManager.instance;
            dynaSvc = fm.getDbProvider().dynaSvc; 
        } 

        if ( item.fileloc == null ) {
            def perSvc = dynaSvc.lookup('PersistenceService', connection); 
            item.fileloc = perSvc.read([_schemaname: 'sys_fileloc', objid: item.filelocid.toString()]); 
        }
        if ( !item.fileloc?.objid ) {
            throw new IllegalStateException("'"+ item.filelocid +"' fileloc record not found");
        }
    }

    public java.net.URL createURL( Map item ) {
        if ( item._validated.toString() == 'true' ) {
            // no validation required 
        } else {
            validateItem( item );
        }
        
        def sb = new StringBuilder();
        def downloadhost = item.fileloc?.info?.downloadhost;
        if ( downloadhost ) {
            def exprVar = new ExprVar( item.fileloc );
            sb.append(exprVar.eval( downloadhost )); 
            if ( !sb.toString().endsWith('/')) {
                sb.append('/'); 
            }
        }
        else {
            sb.append("ftp://");
            sb.append( item.fileloc?.user?.name ).append(":"); 
            sb.append( item.fileloc?.user?.pwd ).append("@"); 
            sb.append( item.fileloc?.url ).append("/"); 
        }

        if ( item.fileloc?.rootdir ) {
            sb.append( item.fileloc.rootdir ).append("/"); 
        }
        if ( item.filedir ) {
            sb.append( item.filedir ).append("/"); 
        }
        sb.append( item.objid ); 
        if ( item.filetype ) { 
            sb.append(".").append( item.filetype ); 
        } 
        return new java.net.URL( sb.toString()); 
    }
    
    public byte[] toByteArray() {
        def url = createURL();
        if ( url == null ) return null; 
        return com.rameses.io.IOStream.toByteArray( url ); 
    }
    
    
    public Object save() {
        def info = open();

        def name = ( info.name ? info.name : info.caption );
        def file = showSaveDialog( info.filetype, name ); 
        if ( !file ) return null; 

        save( file ); 
        return file; 
    }

    public void save( File file ) { 
        def url = createURL(); 
        save( url, file ); 
    }
    
    
    public void remove() {
        def item = getData();
        if ( item == null ) item = open();
        if ( !item ) return; 
        
        def param = [:]; 
        param.item = [ objid: item.objid ];
        getService().remove( param );
    }
}
