package com.rameses.filemgmt.components;

import com.rameses.rcp.annotations.*;
import com.rameses.rcp.common.*;
import com.rameses.osiris2.client.*;
import com.rameses.osiris2.common.*;

import javax.swing.JFileChooser;

class FileViewListPanelModel extends ComponentBean {

    final def fileTypeProvider = com.rameses.filemgmt.FileManager.instance.fileTypeProvider; 
    final Object ITEMS_LOCKED = new Object(); 

    @Binding
    def binding;
    
    AbstractFileViewListModel fileViewListModel; 


    def items;
    def selectedItem;
    
    def listHandler = [
        fetchList: { 
            synchronized( ITEMS_LOCKED ) {
                return items;     
            }
        },
        openItem: { o-> 
            openItemImpl( o ); 
        },
        removeItem: { o-> 
            return removeItemImpl( o ); 
        }
    ] as IconListModel;
    

    void setInitItems( boolean b ) {
        rebuildItems(); 
    }


    void refresh() { 
        rebuildItems(); 
    }

    void addItem() {
        def fileid = fileViewListModel.getFileId(); 
        if ( fileid ) {
            def sysfile = findSysFile( fileid ); 
            fileid = sysfile?.objid; 
        }

        def param = fileViewListModel.createParam();
        if ( param == null ) param = [:]; 

        if ( !param.filetype ) param.filetype = 'jpg'; 

        def filetype = fileTypeProvider.types.find{ it.objid == param.filetype }
        if ( !filetype ) throw new Exception(''+ param.filetype  +' file type not supported');

        def filter = fileViewListModel.createFileFilter( filetype.objid, filetype.title ); 
        fileViewListModel.setFileFilter( filter ); 

        def fileChooser = fileViewListModel.fileChooser; 

        int opt = fileChooser.showOpenDialog( listHandler.binding?.owner ); 
        if ( opt == JFileChooser.APPROVE_OPTION ) { 
            def files = fileChooser.getSelectedFiles(); 
            uploadFiles( files, param, fileid ); 
        } 
    }

    void openItem() {
        openItemImpl( selectedItem ); 
    }

    void openItemImpl( fileitem ) {
        if ( !fileitem ) return; 

        def m = [ fileitem: fileitem ]; 
        m.connection = fileViewListModel.getConnection(); 
        m.fileitem.connection = m.connection;
        Modal.show('sys_fileitem:open', m ); 
    }

    void removeItem() {
        if ( !selectedItem ) return; 

        int idx = listHandler.indexOf( selectedItem ); 
        boolean b = removeItemImpl( selectedItem ); 
        if ( b ) { 
            items.remove( idx ); 
            listHandler.removeItemAt( idx ); 
        } 
    }
    boolean removeItemImpl( fileitem ) {
        if ( !fileitem ) return false;

        if ( !MsgBox.confirm('You are about to remove this item. Proceed?')) return false; 

        def param = fileViewListModel.createParam();
        if ( param == null ) param = [:]; 

        def map = [ item: [objid: fileitem.objid], param: param ]; 
        fileViewListModel.beforeRemoveItem( map ); 

        def connection = fileViewListModel.getConnection(); 
        def dbProvider = com.rameses.filemgmt.FileManager.instance.dbProvider; 
        def res = dbProvider.remove( map, connection ); 
        return ( res.status.toString() == 'success' ); 
    }


    def uploadFiles( def files, Map param, String fileid ) {
        if ( !files ) throw new Exception("Please specify at least one file to upload"); 
        
        def conn = fileViewListModel.getConnection();

        if ( param == null ) param = [:]; 

        def filetype = param.filetype; 
        files.each{ o-> 
            if ( o instanceof java.io.File ) {
                if ( !filetype ) {
                    def sname = o.name.toString().toLowerCase(); 
                    int idx = sname.lastIndexOf('.'); 
                    if ( idx > 0 ) { 
                        filetype = sname.substring( idx+1 ); 
                    }
                    param.filetype = filetype; 
                }
            }
            else {
                throw new Exception("Please specify a valid File object"); 
            }
        }
        
        if ( !filetype ) 
            throw new Exception("Please specify the filetype in the 'param' argument"); 

        def res = null; 

        def map = [:]; 
        map.allowAskFileInfo = false; 
        map.param = param; 
        map.files = files;
        map.fileid = fileid; 
        map.filetype = filetype;
        map.connection = conn; 
        map.handler = { o-> 
            o.remove('items'); 
            o.remove('param'); 
            res = o; 
        } 
        Modal.show("sys_fileupload_task", map ); 
        
        if ( res ) { 
            if ( !fileid ) { 
                afterCreate( res ); 
            }
            
            afterUpload( res ); 
        } 
        else {
            cancelledUpload(); 
        }
        return res; 
    }

    void afterCreate( res ) {
        fileViewListModel.afterCreate( res ); 
    }
    
    void afterUpload( res ) { 
        fileViewListModel.afterUpload( res ); 
        rebuildItems(); 
    }

    void cancelledUpload() {
        fileViewListModel.cancelledUpload(); 
    }


    def findSysFile( objid ) {
        if ( !objid ) return null; 

        def connection = fileViewListModel.getConnection(); 
        def sysfile = com.rameses.filemgmt.FileManager.instance.dbProvider.read([ objid: objid ], connection ); 
        if ( sysfile ) {
            sysfile.items.each{ o-> 
                if ( !o.filetype ) { 
                    o.filetype = sysfile.filetype; 
                }
            }
        }
        return sysfile; 
    }

    void rebuildItems() {
        def fileid = fileViewListModel.getFileId(); 
        def sysfile = findSysFile( fileid ); 

        synchronized( ITEMS_LOCKED ) { 
            if ( items != null ) items.clear(); 

            items = [];
            sysfile?.items.each{ o-> 
                if ( !o ) return;

                def item = listHandler.createItem( o, o.thumbnail, o.caption ); 
                if ( item ) items << item; 
            }
        } 
    }
}
